// History management
const HISTORY_KEY = "resume_analysis_history";
const MAX_HISTORY = 20;

function extractJobName(jobDescription) {
  // Try to get from input field first
  const jobNameInput = document.getElementById("job-name");
  if (jobNameInput && jobNameInput.value.trim()) {
    return jobNameInput.value.trim();
  }

  // Auto-detect from first line of job description
  const lines = jobDescription.trim().split("\n");
  const firstLine = lines[0].trim();
  
  // If first line looks like a job title (under 100 chars, not all caps unless short)
  if (firstLine.length > 0 && firstLine.length < 100) {
    // Remove common prefixes
    let jobName = firstLine
      .replace(/^(Job Title|Position|Role):\s*/i, "")
      .replace(/^About (the|this) (role|position|job):\s*/i, "")
      .trim();
    
    if (jobName.length > 5) {
      return jobName;
    }
  }
  
  return "Unknown Position";
}

function loadHistory() {
  try {
    const history = localStorage.getItem(HISTORY_KEY);
    return history ? JSON.parse(history) : [];
  } catch (e) {
    console.error("Failed to load history:", e);
    return [];
  }
}

function saveToHistory(data) {
  try {
    const history = loadHistory();
    const jobName = extractJobName(jobDescription.value);
    const entry = {
      id: Date.now(),
      timestamp: new Date().toISOString(),
      fileName: selectedFile?.fileName || selectedFile?.name || "Unknown",
      jobName: jobName,
      jobDescriptionPreview: jobDescription.value.substring(0, 100) + "...",
      score: data.initialAnalysis?.score || 0,
      finalScore: data.finalScore?.score || data.initialAnalysis?.score || 0,
      initialAnalysis: data.initialAnalysis,
      tailoredResume: data.tailoredResume,
      finalScoreData: data.finalScore,
      improvement: data.improvement || 0
    };

    history.unshift(entry);

    if (history.length > MAX_HISTORY) {
      history.splice(MAX_HISTORY);
    }

    localStorage.setItem(HISTORY_KEY, JSON.stringify(history));
    displayHistoryList();
  } catch (e) {
    console.error("Failed to save to history:", e);
  }
}

function displayHistoryList() {
  const historyList = document.getElementById("history-list");
  const history = loadHistory();

  if (history.length === 0) {
    historyList.innerHTML = '<p class="history-empty">No history yet</p>';
    return;
  }

  historyList.innerHTML = history
    .map((entry) => {
      const date = new Date(entry.timestamp);
      const dateStr = date.toLocaleDateString("en-US", {
        month: "short",
        day: "numeric",
      });
      const timeStr = date.toLocaleTimeString([], {
        hour: "2-digit",
        minute: "2-digit",
      });

      let scoreClass = "score-low";
      if (entry.score >= 75) scoreClass = "score-high";
      else if (entry.score >= 50) scoreClass = "score-medium";

      return `
      <div class="history-item" onclick="loadHistoryEntry(${entry.id})">
        <div class="history-item-header">
          <span class="history-filename">${entry.jobName || entry.fileName}</span>
          <span class="history-score ${scoreClass}">${entry.score}</span>
        </div>
        <div class="history-date">${dateStr} at ${timeStr}</div>
      </div>
    `;
    })
    .join("");
}

function loadHistoryEntry(id) {
  const history = loadHistory();
  const entry = history.find((e) => e.id === id);

  if (!entry) {
    showError("History entry not found");
    return;
  }
  
  console.log('Loading history entry:', entry);

  // Set the tailored resume content
  tailoredResumeContent = entry.tailoredResume;

  // Adapt history format - handle both old and new formats
  const data = {
    initialAnalysis: entry.initialAnalysis || {
      score: entry.score || 0,
      strengths: [],
      gaps: [],
      explanation: ""
    },
    tailoredResume: entry.tailoredResume,
    finalScore: entry.finalScoreData || entry.finalScore || {
      score: entry.finalScore || entry.score || 0,
      keywordMatchRate: 0,
      atsPassLikelihood: 85
    },
    improvement: entry.improvement || 0
  };

  // Display results without re-saving to history
  displayResults(data, { fromHistory: true });
}

function clearHistory() {
  if (confirm("Are you sure you want to clear all history?")) {
    localStorage.removeItem(HISTORY_KEY);
    displayHistoryList();
  }
}

// DOM Elements
const dropZone = document.getElementById("drop-zone");
const dropZoneContent = document.getElementById("drop-zone-content");
const selectedFileDisplay = document.getElementById("selected-file-display");
const selectedFileName = document.getElementById("selected-file-name");
const changeFileBtn = document.getElementById("change-file-btn");
const jobDescription = document.getElementById("job-description");
const analyzeBtn = document.getElementById("analyze-btn");
const loadingSection = document.getElementById("loading-section");
const resultsSection = document.getElementById("results-section");
const errorToast = document.getElementById("error-toast");
const errorMessage = document.getElementById("error-message");
const previewPanel = document.getElementById("preview-document");

// State
let selectedFile = null;
let tailoredResumeContent = "";

// Live preview - simple HTML rendering for web
async function updateLivePreview() {
  if (!previewPanel || !tailoredResumeContent) return;

  try {
    // Format the resume content as HTML
    const formattedHTML = formatResumeForPreview(tailoredResumeContent);
    previewPanel.innerHTML = formattedHTML;
  } catch (error) {
    console.error("Error generating preview:", error);
  }
}

// Simple HTML escape helper
function escapeHtml(text) {
  const div = document.createElement("div");
  div.textContent = text;
  return div.innerHTML;
}

// Right-click context menu - use browser default for web
// (Electron-specific features removed for web version)

// File Selection
dropZone.addEventListener("click", async (e) => {
  if (e.target === changeFileBtn) return;
  await selectFile();
});

changeFileBtn.addEventListener("click", async (e) => {
  e.stopPropagation();
  await selectFile();
});

async function selectFile() {
  try {
    // Create file input for web
    const fileInput = document.createElement('input');
    fileInput.type = 'file';
    fileInput.accept = '.pdf,.docx';
    fileInput.onchange = (e) => {
      const file = e.target.files[0];
      if (file) {
        selectedFile = {
          file: file,
          fileName: file.name,
          filePath: file.name
        };
        showSelectedFile(file.name);
        validateForm();
      }
    };
    fileInput.click();
  } catch (error) {
    showError("Failed to select file");
  }
}

function showSelectedFile(fileName) {
  dropZoneContent.classList.add("hidden");
  selectedFileDisplay.classList.remove("hidden");
  selectedFileName.textContent = fileName;
  dropZone.style.borderColor = "#10b981";
  dropZone.style.background = "rgba(16, 185, 129, 0.05)";
}

function resetFileSelection() {
  selectedFile = null;
  dropZoneContent.classList.remove("hidden");
  selectedFileDisplay.classList.add("hidden");
  dropZone.style.borderColor = "";
  dropZone.style.background = "";
}

// Form Validation
jobDescription.addEventListener("input", validateForm);

function validateForm() {
  const hasFile = selectedFile !== null;
  const hasJobDescription = jobDescription.value.trim().length > 50;
  analyzeBtn.disabled = !(hasFile && hasJobDescription);
}

// Analyze Resume
analyzeBtn.addEventListener("click", async () => {
  if (!selectedFile || !jobDescription.value.trim()) return;

  // Get selected model
  const modelSelect = document.getElementById("model-select");
  const selectedModel = modelSelect.value;

  // Hide input section and show loading
  document.querySelector(".input-section").style.display = "none";
  resultsSection.classList.add("hidden");
  loadingSection.classList.remove("hidden");

  // Reset loading steps
  resetLoadingSteps();

  try {
    // Step 1: Extracting
    updateLoadingStep(1, "active");
    await delay(500);
    updateLoadingStep(1, "completed");

    // Step 2: Scoring
    updateLoadingStep(2, "active");

    // Get selected model
    const modelSelect = document.getElementById("model-select");
    const selectedModel = modelSelect ? modelSelect.value : 'gpt-4-turbo-preview';

    const formData = new FormData();
    formData.append('resume', selectedFile.file);
    formData.append('jobDescription', jobDescription.value.trim());
    formData.append('model', selectedModel);
    
    const response = await fetch('/api/analyze-electron', {
      method: 'POST',
      body: formData
    });
    
    console.log('Response status:', response.status);

    const contentType = response.headers.get("content-type") || "";
    const isJson = contentType.includes("application/json");

    if (!response.ok) {
      const errorText = await response.text();
      if (response.status === 404) {
        throw new Error(
          "API endpoint not found (404). Make sure the server is running and updated with /api/analyze-electron.",
        );
      }
      throw new Error(
        isJson
          ? (JSON.parse(errorText).error || "Failed to analyze resume")
          : `Server error (${response.status}). ${errorText.slice(0, 200)}`,
      );
    }

    if (!isJson) {
      const text = await response.text();
      throw new Error(
        `Unexpected response from server. Expected JSON but got: ${text.slice(0, 200)}`,
      );
    }

    const result = await response.json();
    console.log('API Response:', result);
    
    // Electron-style response (gold standard)
    const adaptedResult = result;
    
    console.log('Adapted result:', adaptedResult);
    console.log('Score improvement:', adaptedResult.initialAnalysis.score, '->', adaptedResult.finalScore.score);
    console.log('Keyword match:', adaptedResult.finalScore.keywordMatchRate + '%');

    updateLoadingStep(2, "completed");

    // Step 3: Generating
    updateLoadingStep(3, "active");
    await delay(300);
    updateLoadingStep(3, "completed");

    // Step 4: Final scoring
    updateLoadingStep(4, "active");
    await delay(300);
    updateLoadingStep(4, "completed");

    await delay(500);

    // Display results
    displayResults(adaptedResult);
  } catch (error) {
    console.error("Analysis error:", error);
    showError(error.message || "Failed to analyze resume");
    document.querySelector(".input-section").style.display = "block";
  } finally {
    loadingSection.classList.add("hidden");
  }
});

function resetLoadingSteps() {
  for (let i = 1; i <= 4; i++) {
    const step = document.getElementById(`step-${i}`);
    step.classList.remove("active", "completed");
  }
}

function updateLoadingStep(stepNumber, status) {
  const step = document.getElementById(`step-${stepNumber}`);
  step.classList.remove("active", "completed");
  step.classList.add(status);
}

function delay(ms) {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

// Display Results
function displayResults(data, options = {}) {
  const { fromHistory = false } = options;
  const { initialAnalysis, tailoredResume, finalScore } = data;
  
  console.log('displayResults called with:', { initialAnalysis, tailoredResume, finalScore });
  
  // Validate required data
  if (!initialAnalysis || !tailoredResume) {
    console.error('Missing required data in displayResults:', data);
    showError('Invalid data structure. Please analyze the resume again.');
    return;
  }
  
  tailoredResumeContent = tailoredResume;

  // Focus on results view
  const inputSection = document.querySelector(".input-section");
  if (inputSection) inputSection.style.display = "none";

  // Before score
  const beforeScore = initialAnalysis.score || 0;
  const beforeKeywordMatch =
    initialAnalysis.keywordAnalysis?.score ||
    initialAnalysis.keywordMatchRate ||
    0;

  animateScore("before-score-value", beforeScore);
  animateScoreCircle("before-score-circle", beforeScore);
  document.getElementById("before-keyword-match").textContent =
    `${beforeKeywordMatch}%`;

  // After score
  const afterScore = finalScore?.score || beforeScore;
  const afterKeywordMatch =
    finalScore?.keywordAnalysis?.score || finalScore?.keywordMatchRate || beforeKeywordMatch;
  const atsLikelihood = finalScore?.atsPassLikelihood || 0;

  animateScore("after-score-value", afterScore);
  animateScoreCircle("after-score-circle", afterScore);
  document.getElementById("after-keyword-match").textContent =
    `${afterKeywordMatch}%`;
  document.getElementById("ats-likelihood").textContent = `${atsLikelihood}%`;

  // Improvement badge
  const improvement = afterScore - beforeScore;
  const improvementBadge = document.getElementById("improvement-badge");
  improvementBadge.textContent =
    improvement >= 0 ? `+${improvement}` : improvement;
  improvementBadge.style.background = improvement >= 0 ? "#10b981" : "#ef4444";

  // Strengths
  const strengthsList = document.getElementById("strengths-list");
  strengthsList.innerHTML = "";
  (initialAnalysis.strengths || []).forEach((strength) => {
    const li = document.createElement("li");
    li.textContent = strength;
    strengthsList.appendChild(li);
  });

  // Improvements - show what was improved
  const improvementsList = document.getElementById("improvements-list");
  if (improvementsList) {
    improvementsList.innerHTML = "";
    
    // If we have detailed improvements data, show it
    if (data.improvements) {
      const improvements = data.improvements;
      
      // Show sections improved
      if (improvements.sectionsImproved && improvements.sectionsImproved.length > 0) {
        const li = document.createElement("li");
        li.innerHTML = `<strong>Sections Enhanced:</strong> ${improvements.sectionsImproved.join(', ')}`;
        improvementsList.appendChild(li);
      }
      
      // Show how gaps were addressed (NOT keywords)
      if (improvements.gapsAddressed && improvements.gapsAddressed.length > 0) {
        improvements.gapsAddressed.forEach((item) => {
          const li = document.createElement("li");
          // Check if it's an object with gap/howAddressed or just a string
          if (typeof item === 'object' && item.gap) {
            li.innerHTML = `<strong>${item.gap}:</strong> ${item.howAddressed}`;
          } else {
            li.textContent = item;
          }
          improvementsList.appendChild(li);
        });
      }
      
      // Show score improvements
      if (improvements.scoreChange !== undefined) {
        const li = document.createElement("li");
        const change = improvements.scoreChange >= 0 ? `+${improvements.scoreChange}` : improvements.scoreChange;
        li.innerHTML = `<strong>Overall Score:</strong> ${change} points`;
        li.style.color = improvements.scoreChange >= 0 ? '#10b981' : '#ef4444';
        improvementsList.appendChild(li);
      }
    } else {
      // Fallback to showing gaps from initial analysis
      (initialAnalysis.gaps || []).forEach((gap) => {
        const li = document.createElement("li");
        li.textContent = gap;
        improvementsList.appendChild(li);
      });
    }
  }

  // Keywords - show in separate section
  const keywordsContainer = document.getElementById("keywords-container");
  keywordsContainer.innerHTML = "";

  // Use keywordsIntegrated from improvements if available
  const keywordsToShow = data.improvements?.keywordsIntegrated || 
                         data.improvements?.keywordsAdded || 
                         [];
  
  if (keywordsToShow.length > 0) {
    keywordsToShow.forEach((keyword) => {
      const span = document.createElement("span");
      span.className = "keyword-badge";
      span.textContent = keyword;
      keywordsContainer.appendChild(span);
    });
  } else {
    keywordsContainer.innerHTML = '<p class="text-gray-400">No keywords integrated yet</p>';
  }

  // Old keyword display logic (keep for backwards compatibility)
  const matchedKeywords =
    initialAnalysis.keywordAnalysis?.matchedKeywords || [];
  const addedKeywords =
    finalScore.keywordAnalysis?.matchedKeywords?.filter(
      (k) => !matchedKeywords.includes(k),
    ) || [];

  matchedKeywords.forEach((keyword) => {
    const tag = document.createElement("span");
    tag.className = "keyword-tag matched";
    tag.textContent = keyword;
    keywordsContainer.appendChild(tag);
  });

  addedKeywords.forEach((keyword) => {
    const tag = document.createElement("span");
    tag.className = "keyword-tag added";
    tag.textContent = `+ ${keyword}`;
    keywordsContainer.appendChild(tag);
  });

  // Tailored resume
  const resumeElement = document.getElementById("tailored-resume");
  // Convert **text** to <strong>text</strong> for display
  const formattedResume = tailoredResume
    .replace(/\*\*([^*]+)\*\*/g, "<strong>$1</strong>")
    .replace(/\n/g, "<br>");
  resumeElement.innerHTML = formattedResume;

  // Save to history unless we are reloading an existing entry
  if (
    !fromHistory &&
    data.initialAnalysis &&
    data.tailoredResume &&
    data.finalScore
  ) {
    saveToHistory(data);
  }

  // Show results
  resultsSection.classList.remove("hidden");
  resultsSection.scrollIntoView({ behavior: "smooth", block: "start" });

  updateLivePreview();
}

function animateScore(elementId, targetScore) {
  const element = document.getElementById(elementId);
  let current = 0;
  const duration = 1500;
  const increment = targetScore / (duration / 16);

  const animate = () => {
    current += increment;
    if (current < targetScore) {
      element.textContent = Math.round(current);
      requestAnimationFrame(animate);
    } else {
      element.textContent = targetScore;
    }
  };

  animate();
}

function animateScoreCircle(elementId, score) {
  const circle = document.getElementById(elementId);
  const circumference = 283;
  const offset = circumference - (score / 100) * circumference;

  setTimeout(() => {
    circle.style.strokeDashoffset = offset;
  }, 100);
}

// Copy Resume
document.getElementById("copy-btn").addEventListener("click", async () => {
  try {
    await navigator.clipboard.writeText(tailoredResumeContent);
    showToast("Copied to clipboard!", "success");
  } catch (error) {
    showError("Failed to copy to clipboard");
  }
});

// Save PDF - Now using single professional template
document.getElementById("save-pdf-btn").addEventListener("click", async () => {
  try {
    if (!tailoredResumeContent) {
      showError("No resume content to save");
      return;
    }

    // Show saving indicator
    const btn = document.getElementById("save-pdf-btn");
    const originalText = btn.innerHTML;
    btn.innerHTML =
      '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="animate-spin"><circle cx="12" cy="12" r="10" stroke-opacity="0.25"></circle><path d="M12 2a10 10 0 0 1 10 10" stroke-opacity="0.75"></path></svg> Generating PDF...';
    btn.disabled = true;

    // Web version: download as HTML that can be printed to PDF
    const htmlContent = formatResumeAsHTML(tailoredResumeContent);
    const blob = new Blob([htmlContent], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `resume-${Date.now()}.html`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);

    btn.innerHTML = originalText;
    btn.disabled = false;
    showToast("HTML downloaded! Open in browser and use Print > Save as PDF", "success");
  } catch (error) {
    const btn = document.getElementById("save-pdf-btn");
    btn.disabled = false;
    showError("Failed to save PDF: " + (error.message || "Unknown error"));
  }
});

// Save HTML - Download the HTML file that can be opened in browser and printed as PDF
document.getElementById("save-html-btn").addEventListener("click", async () => {
  try {
    if (!tailoredResumeContent) {
      showError("No resume content to save");
      return;
    }

    const btn = document.getElementById("save-html-btn");
    const originalText = btn.innerHTML;
    btn.innerHTML =
      '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="animate-spin"><circle cx="12" cy="12" r="10" stroke-opacity="0.25"></circle><path d="M12 2a10 10 0 0 1 10 10" stroke-opacity="0.75"></path></svg> Generating HTML...';
    btn.disabled = true;

    // Web version: direct download
    const htmlContent = formatResumeAsHTML(tailoredResumeContent);
    const blob = new Blob([htmlContent], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `resume-${Date.now()}.html`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);

    btn.innerHTML = originalText;
    btn.disabled = false;
    showToast("HTML saved successfully!", "success");
  } catch (error) {
    const btn = document.getElementById("save-html-btn");
    btn.disabled = false;
    showError("Failed to save HTML: " + (error.message || "Unknown error"));
  }
});

// Removed Word, Markdown, and Text save handlers - now only using PDF export

// Start Over
document.getElementById("start-over-btn").addEventListener("click", () => {
  resultsSection.classList.add("hidden");
  document.querySelector(".input-section").style.display = "block";
  resetFileSelection();
  jobDescription.value = "";
  validateForm();
  window.scrollTo({ top: 0, behavior: "smooth" });
});

// Error handling
function showError(message) {
  errorMessage.textContent = message;
  errorToast.classList.remove("hidden");
  setTimeout(() => {
    errorToast.classList.add("hidden");
  }, 5000);
}

function showToast(message, type) {
  // Temporarily reuse error toast for success messages
  const toast = errorToast;
  const msgElement = errorMessage;

  if (type === "success") {
    toast.style.background = "#10b981";
  } else {
    toast.style.background = "#ef4444";
  }

  msgElement.textContent = message;
  toast.classList.remove("hidden");

  setTimeout(() => {
    toast.classList.add("hidden");
    toast.style.background = "";
  }, 3000);
}

// Clear history button
document
  .getElementById("clear-history-btn")
  .addEventListener("click", clearHistory);

// Helper function to format resume content for preview/display
function formatResumeForPreview(text) {
  if (!text) return "";
  
  const lines = text.split("\n");
  let html = "";
  let isFirstLine = true;

  for (let line of lines) {
    line = line.trim();
    if (!line) {
      html += "<br>";
      continue;
    }

    // Section headers (all caps)
    if (
      line === line.toUpperCase() &&
      line.length > 5 &&
      !line.match(/^[\d\(\)\-\+\s]+$/)
    ) {
      html += `<h2 style="font-size: 18px; margin-top: 25px; margin-bottom: 10px; color: #2563eb; font-weight: bold; border-bottom: 2px solid #2563eb; padding-bottom: 5px;">${escapeHtml(line)}</h2>`;
    }
    // Bullet points
    else if (line.match(/^[•\-\*]\s/)) {
      const bulletContent = line.substring(2);
      const formatted = convertMarkdownToHTML(bulletContent);
      html += `<div style="margin-left: 20px; margin-bottom: 5px;">• ${formatted}</div>`;
    }
    // Lines with markdown bold
    else if (line.includes("**") || line.includes("*")) {
      const formatted = convertMarkdownToHTML(line);
      // Check if it looks like a job title (has | or dates)
      if (line.match(/\||\d{4}|Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec/)) {
        html += `<p style="font-style: italic; margin: 10px 0 5px 0;">${formatted}</p>`;
      } else {
        html += `<p style="margin: 8px 0;">${formatted}</p>`;
      }
    }
    // Name (first line)
    else if (isFirstLine && line.length < 50 && !line.includes("@") && !line.match(/\d{3}/)) {
      html += `<h1 style="font-size: 28px; margin-bottom: 10px; font-weight: bold; color: #000;">${escapeHtml(line)}</h1>`;
      isFirstLine = false;
    }
    // Regular text
    else {
      html += `<p style="margin: 5px 0;">${escapeHtml(line)}</p>`;
      isFirstLine = false;
    }
  }

  return html;
}

function convertMarkdownToHTML(text) {
  // First escape HTML to prevent XSS
  let escaped = escapeHtml(text);
  // Then convert markdown bold (**text**) to HTML <strong>
  escaped = escaped.replace(/\*\*([^*]+)\*\*/g, "<strong>$1</strong>");
  // Convert markdown italic (*text*) to HTML <em> (but not bullet points)
  escaped = escaped.replace(/(?<!\*)\*([^*]+)\*(?!\*)/g, "<em>$1</em>");
  return escaped;
}

function escapeHtml(text) {
  const div = document.createElement("div");
  div.textContent = text;
  return div.innerHTML;
}

// Helper function to format resume as HTML for download
function formatResumeAsHTML(content) {
  return `
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Resume</title>
  <style>
    body {
      font-family: Arial, Helvetica, sans-serif;
      line-height: 1.6;
      max-width: 800px;
      margin: 40px auto;
      padding: 20px;
      color: #333;
    }
    h1 {
      font-size: 28px;
      margin-bottom: 10px;
      font-weight: bold;
      color: #000;
    }
    h2 {
      font-size: 18px;
      margin-top: 25px;
      margin-bottom: 10px;
      color: #2563eb;
      font-weight: bold;
      border-bottom: 2px solid #2563eb;
      padding-bottom: 5px;
    }
    p {
      margin: 8px 0;
    }
    .contact-info {
      margin-bottom: 20px;
      font-size: 14px;
    }
    @media print {
      body {
        margin: 0;
        padding: 20px;
      }
    }
  </style>
</head>
<body>
${formatResumeForPreview(content)}
</body>
</html>`;
}

