require('dotenv').config();
const express = require('express');
const multer = require('multer');
const path = require('path');
const fs = require('fs').promises;
const cors = require('cors');
const OpenAI = require('openai');
const pdfParse = require('pdf-parse');
const mammoth = require('mammoth');
const rateLimit = require('express-rate-limit');

const app = express();
const PORT = process.env.PORT || 3000;

// Initialize OpenAI
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY
});

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.static('public'));

// Rate limiting for API endpoints
const apiLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 10, // Limit each IP to 10 requests per windowMs
  message: 'Too many requests from this IP, please try again later.',
  standardHeaders: true,
  legacyHeaders: false,
});

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: async (req, file, cb) => {
    const dir = './uploads';
    try {
      await fs.mkdir(dir, { recursive: true });
      cb(null, dir);
    } catch (err) {
      console.error('Error creating uploads directory:', err);
      cb(err);
    }
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + '-' + file.originalname);
  }
});

const upload = multer({
  storage: storage,
  fileFilter: (req, file, cb) => {
    const ext = path.extname(file.originalname).toLowerCase();
    if (ext !== '.pdf' && ext !== '.docx') {
      return cb(new Error('Only PDF and DOCX files are allowed'));
    }
    cb(null, true);
  }
});

// Extract text from PDF
async function extractTextFromPDF(filePath) {
  const dataBuffer = await fs.readFile(filePath);
  const data = await pdfParse(dataBuffer);
  return data.text;
}

// Extract text from DOCX
async function extractTextFromDOCX(filePath) {
  const buffer = await fs.readFile(filePath);
  const result = await mammoth.extractRawText({ buffer: buffer });
  return result.value;
}

// Extract text based on file type
async function extractText(filePath) {
  const ext = path.extname(filePath).toLowerCase();
  if (ext === '.pdf') {
    return await extractTextFromPDF(filePath);
  } else if (ext === '.docx') {
    return await extractTextFromDOCX(filePath);
  }
  throw new Error('Unsupported file type');
}

// Electron-style keyword extraction
function extractKeywordsElectron(jobDescription) {
  const skillPatterns = [
    /\b(python|javascript|typescript|java|c\+\+|c#|ruby|go|rust|swift|kotlin|php|scala|r|sql|html|css|react|angular|vue|node\.?js|express|django|flask|spring|\.net|aws|azure|gcp|docker|kubernetes|git|agile|scrum|ci\/cd|machine learning|deep learning|ai|data science|analytics|tableau|power bi|excel|sap|salesforce|jira|confluence)\b/gi,
    /\b(leadership|communication|teamwork|problem.?solving|analytical|strategic|project management|stakeholder|collaboration|presentation|negotiation|decision.?making|time management|organizational|interpersonal|creative|innovative|detail.?oriented|self.?motivated|proactive)\b/gi,
    /\b(bachelor|master|mba|phd|certified|certification|pmp|cpa|cfa|aws certified|google certified|microsoft certified|scrum master|six sigma)\b/gi,
    /\b(\d+\+?\s*years?\s*(of\s*)?(experience|exp))\b/gi,
  ];

  const keywords = new Set();
  skillPatterns.forEach((pattern) => {
    const matches = jobDescription.match(pattern);
    if (matches) {
      matches.forEach((match) => keywords.add(match.toLowerCase().trim()));
    }
  });

  return Array.from(keywords);
}

function calculateKeywordScoreElectron(text, keywords) {
  const textLower = text.toLowerCase();
  const matchedKeywords = [];
  const missingKeywords = [];

  keywords.forEach((keyword) => {
    if (textLower.includes(keyword.toLowerCase())) {
      matchedKeywords.push(keyword);
    } else {
      missingKeywords.push(keyword);
    }
  });

  const score =
    keywords.length > 0
      ? Math.round((matchedKeywords.length / keywords.length) * 100)
      : 0;

  return {
    score,
    matchedKeywords,
    missingKeywords,
    totalKeywords: keywords.length,
  };
}

// Electron-style analysis (gold standard)
async function analyzeResumeFitElectron(
  resumeText,
  jobDescription,
  model = "gpt-4-turbo-preview",
) {
  try {
    const keywords = extractKeywordsElectron(jobDescription);
    const keywordAnalysis = calculateKeywordScoreElectron(resumeText, keywords);

    const prompt = `You are an expert HR professional, ATS (Applicant Tracking System) specialist, and resume analyst. Analyze the following resume against the job description comprehensively.

RESUME:
${resumeText}

JOB DESCRIPTION:
${jobDescription}

EXTRACTED KEYWORDS FROM JOB: ${keywords.join(", ")}
KEYWORD MATCH RATE: ${keywordAnalysis.score}%
MATCHED KEYWORDS: ${keywordAnalysis.matchedKeywords.join(", ")}
MISSING KEYWORDS: ${keywordAnalysis.missingKeywords.join(", ")}

Provide a detailed analysis including:
1. An overall fit score from 0-100 (consider both qualitative fit AND keyword matching - ATS systems heavily weight keyword matching)
2. A detailed explanation of the score
3. Key strengths that align with the job requirements
4. Critical gaps that need to be addressed
5. Specific keywords/skills from the job description that are missing or underrepresented
6. ATS compatibility assessment

Respond in JSON format:
{
  "score": <number>,
  "explanation": "<detailed string explaining the score>",
  "strengths": ["<specific strength 1>", "<specific strength 2>", ...],
  "gaps": ["<specific gap 1>", "<specific gap 2>", ...],
  "missingKeywords": ["<keyword1>", "<keyword2>", ...],
  "atsCompatibility": "<assessment of how well resume will perform in ATS>",
  "keywordMatchRate": <number>
}`;

    const completion = await openai.chat.completions.create({
      model: model,
      messages: [{ role: "user", content: prompt }],
      temperature: 0.5,
      response_format: { type: "json_object" },
    });

    const result = JSON.parse(completion.choices[0].message.content);
    result.extractedKeywords = keywords;
    result.keywordAnalysis = keywordAnalysis;

    return result;
  } catch (error) {
    console.error("Error analyzing resume fit (electron):", error);
    if (
      error.message?.includes("content_policy") ||
      error.message?.includes("safety")
    ) {
      throw new Error(
        "Content was flagged by OpenAI's safety system. Please ensure your resume and job description contain professional content only.",
      );
    }
    throw error;
  }
}

// Electron-style tailored resume generation
async function generateTailoredResumeElectron(
  resumeText,
  jobDescription,
  analysisResult,
  model = "gpt-4-turbo-preview",
) {
  try {
    const prompt = `You are an elite professional resume writer with 20+ years of experience crafting resumes that pass ATS systems and impress hiring managers. Your task is to create a COMPREHENSIVE, DETAILED, and HIGHLY TARGETED resume.

ORIGINAL RESUME:
${resumeText}

TARGET JOB DESCRIPTION:
${jobDescription}

INITIAL ANALYSIS:
- Current Score: ${analysisResult.score}/100
- Strengths: ${analysisResult.strengths.join(", ")}
- Gaps to Address: ${analysisResult.gaps.join(", ")}
- Missing Keywords to Include: ${analysisResult.missingKeywords?.join(", ") || "N/A"}
- All Job Keywords: ${analysisResult.extractedKeywords?.join(", ") || "N/A"}

CREATE A TRANSFORMED RESUME FOLLOWING THESE CRITICAL GUIDELINES:

1. **KEYWORD OPTIMIZATION**: 
   - Naturally integrate ALL missing keywords throughout the resume
   - Use exact phrases from the job description where appropriate
   - Include keywords in context (not just listed)
   - Ensure keywords appear in: Summary, Experience bullets, Skills section

2. **PROFESSIONAL SUMMARY** (4-5 powerful sentences):
   - Lead with years of experience and primary expertise area
   - Include 3-4 key skills mentioned in the job description
   - Mention a quantifiable achievement
   - Express alignment with the company's needs

3. **EXPERIENCE SECTION** - For EACH role, provide:
   - 5-7 detailed bullet points (not 2-3)
   - Start each bullet with a strong action verb
   - Include SPECIFIC metrics, percentages, dollar amounts, team sizes
   - **BOLD** the key skills/keywords within descriptions using **asterisks**
   - Show progression and increasing responsibility
   - Tie achievements directly to job requirements

4. **SKILLS SECTION**:
   - Organize into categories (Technical Skills, Soft Skills, Tools, Certifications)
   - Include ALL relevant keywords from the job description
   - List most relevant skills first

5. **ACHIEVEMENTS/PROJECTS** (if applicable):
   - Add a dedicated section for major accomplishments
   - Quantify impact wherever possible

6. **EDUCATION & CERTIFICATIONS**:
   - Include relevant coursework if applicable
   - List certifications that match job requirements

FORMAT REQUIREMENTS:
- Use **bold** for key terms, skills, and metrics that match the job description
- Make the resume 2-3 pages to be comprehensive (not a brief 1-pager)
- Use consistent formatting throughout
- Include clear section headers

CRITICAL INSTRUCTIONS:
- Output ONLY the resume content itself
- DO NOT include any explanatory text, disclaimers, or commentary
- DO NOT say things like "Here is the resume:" or "Due to complexity..." or "This is an outline..."
- DO NOT mention limitations or suggest further edits needed
- Start directly with the candidate's name or professional summary
- End with the last section (Education/Certifications)
- Provide a COMPLETE, FINISHED resume that is ready to use

The goal is to create a resume that will score 85+ when analyzed against this job description.`;

    const completion = await openai.chat.completions.create({
      model: model,
      messages: [
        {
          role: "system",
          content:
            "You are a professional resume writer. Output only the resume content with no explanations, disclaimers, or commentary. Never include phrases like 'Here is your resume' or 'Due to complexity'. Start directly with the resume.",
        },
        { role: "user", content: prompt },
      ],
      temperature: 0.7,
      max_tokens: 4000,
    });

    let resumeContent = completion.choices[0].message.content.trim();

    const conversationalPhrases = [
      /^(Here is|Here's|Below is|I've created|I've drafted|I will draft|Given the|Due to the complexity).*?[\n\r]+/i,
      /^.*?(resume|outline|template).*?follows?:?\s*[\n\r]+/i,
      /^.*?(However|Note that|Please note|Keep in mind).*?[\n\r]+/i,
      /For a complete.*?[\n\r]+/gi,
      /it's essential to.*?[\n\r]+/gi,
      /you can then expand.*?[\n\r]+/gi,
      /This (resume|outline|template).*?[\n\r]+/gi,
    ];

    conversationalPhrases.forEach((pattern) => {
      resumeContent = resumeContent.replace(pattern, "");
    });

    return resumeContent.trim();
  } catch (error) {
    console.error("Error generating tailored resume (electron):", error);
    if (
      error.message?.includes("content_policy") ||
      error.message?.includes("safety")
    ) {
      throw new Error(
        "Content was flagged by OpenAI's safety system. Please ensure your resume and job description contain professional content only.",
      );
    }
    throw error;
  }
}

// Electron-style scoring of generated resume
async function scoreGeneratedResumeElectron(
  generatedResume,
  jobDescription,
  keywords,
  model = "gpt-4-turbo-preview",
) {
  try {
    const keywordAnalysis = calculateKeywordScoreElectron(generatedResume, keywords);

    const prompt = `You are an expert HR professional and ATS specialist. Score this GENERATED resume against the job description.

GENERATED RESUME:
${generatedResume}

JOB DESCRIPTION:
${jobDescription}

KEYWORD MATCH ANALYSIS:
- Match Rate: ${keywordAnalysis.score}%
- Matched Keywords: ${keywordAnalysis.matchedKeywords.join(", ")}
- Still Missing: ${keywordAnalysis.missingKeywords.join(", ")}

Evaluate how well this tailored resume now matches the job requirements. Provide:
1. A new fit score from 0-100
2. Assessment of improvements made
3. Any remaining gaps
4. ATS pass likelihood (percentage estimate)

Respond in JSON format:
{
  "score": <number>,
  "improvements": ["<improvement 1>", "<improvement 2>", ...],
  "remainingGaps": ["<gap 1>", "<gap 2>", ...],
  "atsPassLikelihood": <percentage>,
  "summary": "<brief summary of the transformation>",
  "keywordMatchRate": <number>
}`;

    const completion = await openai.chat.completions.create({
      model: model,
      messages: [{ role: "user", content: prompt }],
      temperature: 0.5,
      response_format: { type: "json_object" },
    });

    const result = JSON.parse(completion.choices[0].message.content);
    result.keywordAnalysis = keywordAnalysis;

    return result;
  } catch (error) {
    console.error("Error scoring generated resume (electron):", error);

    if (
      error.message?.includes("content_policy") ||
      error.message?.includes("safety")
    ) {
      throw new Error(
        "The content was flagged by our safety system. Please ensure your resume and job description contain only professional information.",
      );
    }

    throw error;
  }
}

// Analyze resume fit and generate score
async function analyzeResumeFit(resumeText, jobDescription) {
  try {
    const prompt = `You are an expert ATS (Applicant Tracking System) analyst and HR professional specializing in resume optimization.

YOUR PRIMARY FOCUS: Evaluate ATS compatibility and keyword matching accuracy.

Resume:
${resumeText}

Job Description:
${jobDescription}

ANALYSIS REQUIREMENTS:
1. KEYWORD MATCH RATE (0-100): Calculate what percentage of important keywords from the job description appear in the resume
   - Count technical skills, tools, methodologies, certifications
   - Count soft skills and competencies mentioned
   - Be strict but fair in matching (variations count: e.g., "JS" matches "JavaScript")
   
2. ATS COMPATIBILITY (0-100): Evaluate how well the resume will perform in ATS systems
   - Standard section headers (Summary, Experience, Skills, Education) = good
   - Simple formatting, no tables/graphics = good
   - Clear dates and job titles = good
   - Keywords in context = good
   - Complex formatting, headers/footers, text boxes = bad
   
3. OVERALL FIT SCORE (0-100): General match between candidate and role

4. MISSING KEYWORDS: List 5-15 specific, important keywords from the job description that are NOT in the resume

5. STRENGTHS: What matches well (2-4 items)

6. GAPS: What's missing or weak (2-4 items)

Respond in JSON format:
{
  "score": <number>,
  "keywordMatchRate": <number>,
  "atsCompatibility": <number>,
  "explanation": "<string focusing on ATS and keyword performance>",
  "strengths": ["<string>", ...],
  "gaps": ["<string>", ...],
  "missingKeywords": ["<keyword>", ...]
}`;

    const completion = await openai.chat.completions.create({
      model: "gpt-3.5-turbo",
      messages: [{ role: "user", content: prompt }],
      temperature: 0.5,
      response_format: { type: "json_object" }
    });

    return JSON.parse(completion.choices[0].message.content);
  } catch (error) {
    console.error('Error analyzing resume fit:', error);
    throw error;
  }
}

// Generate tailored resume
async function generateTailoredResume(resumeText, jobDescription, analysisResult) {
  try {
    const targetKeywordRate = Math.min(95, analysisResult.keywordMatchRate + 30);
    const targetATS = Math.min(98, analysisResult.atsCompatibility + 25);
    const originalLength = resumeText.length;
    const minLength = Math.floor(originalLength * 0.9);
    
    const prompt = `You are an expert ATS (Applicant Tracking System) optimization specialist and resume writer.

⚠️ PRIMARY MISSION: MAXIMIZE ATS COMPATIBILITY AND KEYWORD MATCHING ⚠️

CURRENT RESUME (${originalLength} characters):
${resumeText}

TARGET JOB DESCRIPTION:
${jobDescription}

CURRENT PERFORMANCE:
🎯 KEYWORD MATCH: ${analysisResult.keywordMatchRate}% → TARGET: ${targetKeywordRate}% (PRIORITY #1)
🎯 ATS COMPATIBILITY: ${analysisResult.atsCompatibility}% → TARGET: ${targetATS}% (PRIORITY #2)
- Overall Score: ${analysisResult.score}/100
- MISSING KEYWORDS: ${analysisResult.missingKeywords.join(', ')}
- Gaps: ${analysisResult.gaps.join(', ')}

🔴 CRITICAL SUCCESS CRITERIA (MUST ACHIEVE):
1. LENGTH REQUIREMENT: The improved resume MUST be at least ${minLength} characters (90% of original) or LONGER
2. KEYWORD MATCH RATE: Increase to ${targetKeywordRate}% or higher by integrating ALL missing keywords
3. ATS COMPATIBILITY: Increase to ${targetATS}% or higher with ATS-friendly formatting
4. Every single keyword from the missing list MUST appear at least once in the resume
5. Use keyword variations and related terms to maximize matches (e.g., "JavaScript/JS", "management/managing")
6. DO NOT DELETE OR REMOVE any content - only ADD and ENHANCE

📋 ATS OPTIMIZATION REQUIREMENTS:
- Use standard section headers: Summary, Experience, Skills, Education
- No tables, text boxes, images, or graphics
- No headers/footers
- Use simple bullet points (• or -)
- Include exact keywords from job description
- Use industry-standard job titles
- Add clear contact information at top
- Use common fonts and simple formatting

🔑 KEYWORD INTEGRATION STRATEGY:
Priority 1: Add ALL missing keywords to relevant sections:
  - Professional Summary: Integrate 3-5 keywords naturally
  - Work Experience: Embed keywords in accomplishment bullets
  - Skills Section: List technical keywords explicitly
  - Throughout: Use keyword variations and synonyms

Priority 2: Repeat high-value keywords 2-3 times throughout the resume
Priority 3: Mirror exact phrases from the job description where applicable

✅ QUALITY STANDARDS:
- Keywords must flow naturally in context
- Maintain professional tone and truthfulness
- Focus on quantifiable achievements
- NEVER delete or shorten content - only expand and enhance
- Keep original structure unless ATS-incompatible
- Add more detail, accomplishments, and keywords to meet length requirement

⚠️ LENGTH VERIFICATION:
Your output MUST be at least ${minLength} characters long (currently ${originalLength} characters).
If you're under the minimum, add:
- More detailed accomplishment bullets with keywords
- Expanded professional summary with more keywords
- Additional relevant skills and certifications
- More specific metrics and achievements

CRITICAL OUTPUT REQUIREMENT:
Return ONLY the improved resume text. Do NOT include:
- The job description
- Any analysis or commentary
- Any introductory text like "Here is the improved resume"
- Any closing remarks

Start directly with the candidate's name/contact information and end with the last line of the resume.`;

    const completion = await openai.chat.completions.create({
      model: "gpt-3.5-turbo",
      messages: [{ role: "user", content: prompt }],
      temperature: 0.5,
      max_tokens: 3000
    });

    return completion.choices[0].message.content;
  } catch (error) {
    console.error('Error generating tailored resume:', error);
    throw error;
  }
}

// Analyze what changes were made to the resume
async function analyzeImprovements(originalResume, tailoredResume, initialAnalysis, finalAnalysis) {
  try {
    const prompt = `Compare the original and tailored resume to identify specific improvements made.

MISSING KEYWORDS (what we tried to add): ${initialAnalysis.missingKeywords.join(', ')}
GAPS IDENTIFIED: ${initialAnalysis.gaps.join(', ')}

Analyze what changed:
1. Which resume sections were enhanced (e.g., Summary, Experience, Skills, Education)
2. Which specific keywords from the missing list were successfully integrated into the tailored resume
3. What content improvements were made (without listing keywords again)

Respond in JSON format:
{
  "sectionsImproved": ["<section name>", ...],
  "keywordsIntegrated": ["<keyword from missing list>", ...],
  "gapsAddressed": ["<description of improvement>", ...]
}

Note: keywordsIntegrated should ONLY contain keywords from the missing list that now appear in the resume.
Note: gapsAddressed should describe content improvements, NOT repeat the keywords.`;

    const completion = await openai.chat.completions.create({
      model: "gpt-3.5-turbo",
      messages: [{ role: "user", content: prompt }],
      temperature: 0.5,
      response_format: { type: "json_object" }
    });

    const result = JSON.parse(completion.choices[0].message.content);
    
    // Normalize the response to use keywordsIntegrated
    return {
      sectionsImproved: result.sectionsImproved || [],
      keywordsIntegrated: result.keywordsIntegrated || result.keywordsAdded || [],
      gapsAddressed: result.gapsAddressed || []
    };
  } catch (error) {
    console.error('Error analyzing improvements:', error);
    // Return default structure if analysis fails
    return {
      sectionsImproved: ["Summary", "Experience", "Skills"],
      keywordsIntegrated: initialAnalysis.missingKeywords.slice(0, 5),
      gapsAddressed: ["Enhanced descriptions with quantifiable achievements", "Strengthened alignment with job requirements"]
    };
  }
}

// Electron-style API for Chrome extension (gold standard)
app.post('/api/analyze-electron', apiLimiter, upload.single('resume'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: 'No resume file uploaded' });
    }

    if (!req.body.jobDescription) {
      return res.status(400).json({ error: 'Job description is required' });
    }

    const model = req.body.model || 'gpt-4-turbo-preview';

    // Extract text from resume
    const resumeText = await extractText(req.file.path);

    // Analyze initial resume fit (electron-style)
    const initialAnalysis = await analyzeResumeFitElectron(
      resumeText,
      req.body.jobDescription,
      model
    );

    // Generate tailored resume (electron-style)
    const tailoredResume = await generateTailoredResumeElectron(
      resumeText,
      req.body.jobDescription,
      initialAnalysis,
      model
    );

    // Score the generated resume (electron-style)
    const finalScore = await scoreGeneratedResumeElectron(
      tailoredResume,
      req.body.jobDescription,
      initialAnalysis.extractedKeywords || [],
      model
    );

    // Clean up uploaded file
    await fs.unlink(req.file.path);

    res.json({
      success: true,
      initialAnalysis,
      tailoredResume,
      finalScore,
    });
  } catch (error) {
    console.error('Error in /api/analyze-electron:', error);

    if (req.file) {
      await fs.unlink(req.file.path).catch(err =>
        console.error('Error deleting file:', err)
      );
    }

    res.status(500).json({
      error: error.message || 'Failed to analyze resume. Please try again.'
    });
  }
});

// API endpoint to analyze resume
app.post('/api/analyze', apiLimiter, upload.single('resume'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: 'No resume file uploaded' });
    }

    if (!req.body.jobDescription) {
      return res.status(400).json({ error: 'No job description provided' });
    }

    console.log('Starting resume analysis...');

    // Extract text from uploaded resume
    const resumeText = await extractText(req.file.path);
    console.log('Resume text extracted, length:', resumeText.length);

    // Step 1: Analyze original resume fit (BEFORE score)
    console.log('Analyzing original resume...');
    const initialAnalysis = await analyzeResumeFit(resumeText, req.body.jobDescription);
    console.log('Initial analysis complete, score:', initialAnalysis.score);

    // Step 2: Generate tailored resume
    console.log('Generating tailored resume...');
    const tailoredResume = await generateTailoredResume(
      resumeText,
      req.body.jobDescription,
      initialAnalysis
    );
    console.log('Tailored resume generated, length:', tailoredResume.length);

    // Step 3: Analyze the tailored resume (AFTER score)
    console.log('Analyzing tailored resume...');
    const finalAnalysis = await analyzeResumeFit(tailoredResume, req.body.jobDescription);
    console.log('Final analysis complete, score:', finalAnalysis.score);

    // Step 4: Analyze what improvements were made
    console.log('Analyzing improvements...');
    const improvements = await analyzeImprovements(
      resumeText,
      tailoredResume,
      initialAnalysis,
      finalAnalysis
    );
    console.log('Improvements analyzed:', improvements);

    // Clean up uploaded file
    await fs.unlink(req.file.path);

    // Calculate improvement
    const scoreImprovement = finalAnalysis.score - initialAnalysis.score;
    const keywordImprovement = (finalAnalysis.keywordMatchRate || 0) - (initialAnalysis.keywordMatchRate || 0);
    const atsImprovement = (finalAnalysis.atsCompatibility || 0) - (initialAnalysis.atsCompatibility || 0);
    
    console.log('Score improvement:', scoreImprovement);
    console.log('Keyword improvement:', keywordImprovement);
    console.log('ATS improvement:', atsImprovement);

    // Send response with both before and after scores
    res.json({
      success: true,
      initialAnalysis: {
        score: initialAnalysis.score,
        explanation: initialAnalysis.explanation,
        strengths: initialAnalysis.strengths,
        gaps: initialAnalysis.gaps,
        keywordMatchRate: initialAnalysis.keywordMatchRate || 0,
        atsCompatibility: initialAnalysis.atsCompatibility || 0,
        missingKeywords: initialAnalysis.missingKeywords || []
      },
      tailoredResume: tailoredResume,
      finalAnalysis: {
        score: finalAnalysis.score,
        explanation: finalAnalysis.explanation,
        strengths: finalAnalysis.strengths,
        gaps: finalAnalysis.gaps,
        keywordMatchRate: finalAnalysis.keywordMatchRate || 0,
        atsPassLikelihood: Math.min(98, finalAnalysis.atsCompatibility + 5),
      },
      improvements: {
        scoreChange: scoreImprovement,
        keywordChange: keywordImprovement,
        atsChange: atsImprovement,
        sectionsImproved: improvements.sectionsImproved,
        keywordsIntegrated: improvements.keywordsIntegrated,
        gapsAddressed: improvements.gapsAddressed
      }
    });
  } catch (error) {
    console.error('Error processing request:', error);
    
    // Clean up uploaded file if it exists
    if (req.file) {
      await fs.unlink(req.file.path).catch(err => 
        console.error('Error deleting file:', err)
      );
    }
    
    res.status(500).json({ 
      error: 'Error processing resume',
      message: error.message 
    });
  }
});

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({ status: 'ok' });
});

// Start server
app.listen(PORT, () => {
  console.log(`Server running on http://localhost:${PORT}`);
});
